/************************************************************************
 * NAME:	fstool
 *
 * DESCR:	Decodes and manages a File System (FS) on the given
 *		floppy image.  Currently, this thing doesn't try to guess
 *		the FS, you must tell it which one to do.
 *
 *		See Usage at bottom.
 *
 *		Output is only in SVD format.  Sorry, if you want to 
 *		create a DMK or JV image, you're going to have to write
 *		the dump routines for those.  Shouldn't be too hard.
 *
 * NOTES:  need to appropriately guess and load SVD format!
 *
 *	   Should probably simplify this thing, so that it doesn't have
 *	   to figure out the input format of the floppy.  Would make the
 *	   caller have to decode much less...you know, like incorrectly
 *	   specified image files.
 *
 *	   The question now is, what is an FS structure...something
 *	   parallel to floppy image, or part of it?  A floppy image "has"
 *         an FS as part of it, but the FS really isn't in the floppy in
 *         that there is a ton of other information.  Since the FS's are
 *	   all different, maybe it should be such that each FS structure
 *	   is part of a union/class and has equiv calls for the three
 *	   basic operations:  report, add, delete.
 ************************************************************************/

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <stdlib.h>
#include <fcntl.h>

#include "standard.h"
#include "error.h"
#include "floppy.h"
#include "fssupport.h"

/************************************************************************/
/*		  RELEASE VERSION NUMBER GOES HERE!			*/
/************************************************************************/

#define VERSION	"2.2"

main(argc,argv)
int	argc;
char	*argv[];
{
    int			i;
    int			ch;
    int			fd;
    struct floppy	floppy;
    struct fs		fs;
    int			input_file = 0;		/* stdin is the default */
    char		*input_filename = NULL;
    int			no_effect = FALSE;	/* don't generate svd	*/
    int			verbose = FALSE;	/* level of verbosity	*/
    int			terse = FALSE;		/* level of verbosity	*/
    int			report = FALSE;		/* report disk params	*/
    int			machineform = FALSE;	/* machine readable form*/
    int			add_files = FALSE;
    int			del_files = FALSE;
    int			extract_files = FALSE;
    char		*fs_type = fs_types[0];
    int			list_types = FALSE;
    char		*format_id = NULL;
    char		*output_format_id = NULL;
    int			compress = FALSE;
    int			 input_format_query = FALSE;
    int			 output_format_query = FALSE;

  (void)floppy_init(&floppy);

  floppy_format_init(); 	/* initialize all supported disk formats */

  opterr = 0;

  for (;;) {
    ch = getopt(argc, argv, "?f:F:s:rADEvtnmVlco:");
    if (ch == -1) break;

    switch (ch) {
    case 'F':
	fs_type = argv[optind-1];
	for (i=0; i < fs_count; i++) {
	    if (strcmp(fs_types[i],fs_type) == 0) {
		break;
	    }
	}
	if (i==fs_count) {
	    fprintf(stderr,"Bad os-type \"%s\"\n",fs_type);
	    exit(2);
	}
	break;
	
    case 'A':
	if (del_files || extract_files) {	/* only one can be done	*/
	    usage(argv[0],fs_type);
	    exit(2);
	}
	add_files = TRUE;
	break;
    case 'D':
	if (add_files || extract_files) {	/* only one can be done	*/
	    usage(argv[0],fs_type);
	    exit(2);
	}
	del_files = TRUE;
	break;
    case 'E':
	if (add_files || del_files) {		/* only one can be done	*/
	    usage(argv[0],fs_type);
	    exit(2);
	}
	extract_files = TRUE;
	break;
    case 'r':
	report = TRUE;
	no_effect = TRUE;
	break;
    case 'v':
	/* may want to turn this into a number	*/
	verbose = TRUE;
	break;
    case 't':
	terse = TRUE;
	break;
    case 'm':
	machineform = TRUE;
	break;
    case 'l':
	list_types = TRUE;
	break;
    case 's':
	format_id = floppy_format_list_select(argv[optind-1],FALSE);
	if (format_id == NULL) {
	    fprintf(stderr,"Format type \"%s\" unknown.\n",argv[optind-1]);
	    usage(argv[0],fs_type);
	    exit(3);
	}
	break;
    case 'n':
	no_effect = TRUE;
	break;
    case 'c':
	compress = TRUE;
	break;
    case 'f':
	if (strcmp(argv[optind-1],"-") == 0) {	/* just like not specifying */
	    input_file = 0;
	} else {
	    input_filename = argv[optind-1];
	    input_file = open(input_filename,O_RDONLY);
	    if (input_file == -1) {
		fprintf(stderr,"%s: can't open OS image file \"%s\"\n",argv[0],argv[optind-1]);
		exit(3);
	    }
	}
	break;
    case 'o':
	output_format_id = floppy_format_list_select(argv[optind-1],TRUE);
	if (output_format_id == NULL) {
	    fprintf(stderr,"%s: -o format type \"%s\" unknown or not supported.\n",argv[0],argv[optind-1]);
	    usage(argv[0]);
	    exit(3);
	}
	break;
    case 'V':
	printf("%s: version %s (%s)\n",argv[0],VERSION,COMPDATE);
	exit(0);
    case '?':
	usage(argv[0],fs_type);
	exit(0);
    default:
	usage(argv[0],fs_type);
	exit(2);
    }
  }

  if (optind <= argc - 1) {
      /* have some files...an error if A, D or E NOT spec'd */
      if (!add_files && !del_files && !extract_files) {
	  usage(argv[0],fs_type);
	  exit(2);
      }
  } else {
      /* this is only an error if A, D or E were spec'd	*/
      if (add_files || del_files || extract_files) {
	  usage(argv[0],fs_type);
	  exit(2);
      }	  
  }

  if (list_types) {
      int i;

      for (i=0; i < fs_count; i++) {
	  if (machineform) {
	      printf("%s\t%s (%s)\n",fs_types[i],fs_description(fs_types[i]),fs_types[i]);
	  } else if (verbose) {
	      printf("%s (%s)\n",fs_description(fs_types[i]),fs_types[i]);
	  } else  {
	      printf("%s\n",fs_types[i]);
	  }
      }
      return(0);
  }



  /* run a quick check on the incoming files to make sure that	*/
  /* they are assessible.  This is a nice thing to do so that	*/
  /* bad file names aren't masked by other errors.		*/

  if (add_files) {
      int	bad_oh_bad = FALSE;
      int	fd;

      for( i=optind; i <= argc-1; i++) {
	  if ((fd=open(argv[i],O_RDONLY)) < 0) {
	      if (!bad_oh_bad) {
		  fprintf(stderr,"Can't access file(s): ");
	      } else {
		  fprintf(stderr,", ");
	      }
	      bad_oh_bad = TRUE;
	      fprintf(stderr,"%s",argv[i]);
	  } else {
	      close(fd);
	  }
	  if (bad_oh_bad) {
	      fprintf(stderr,"\n");
	      exit(2);
	  }
      }
  }

  if (!read_image(input_file, &floppy, format_id, argv[0], input_filename)) {
      fprintf(stderr,"%s: couldn't read OS image file\n",argv[0]);
      exit(3);
  }

  if (!fs_init(&floppy,&fs,fs_type)) {
      fprintf(stderr,"%s: floppy format initialization failure\n",argv[0]);
      exit(3);
  }

  /* check it for its FS characteristics	*/
  /* and try to guess which one it is		*/

      /* LATER */

/*  if (fs_check(&floppy) != 0) {
/*      /* error */
/*  }
********/

  if (add_files || del_files || extract_files) {
      /* now process the incoming files for real, they really	*/
      /* should be accessible, but will check them anyway.		*/

      for( i=optind; i <= argc-1; i++) {

	  if (add_files) {
	      if (!fs_add(&fs,argv[i],0)) {
		  fprintf(stderr,"%s: unable to add %s, file exists on image or no space left.\n",argv[0],argv[i]);
		  exit(1);
	      }
	  }
	  if (del_files) {
	      if (fs_del(&fs,argv[i]) != 0) {
		  /* can't del it...not there?  other error?	*/
		  /* error or maybe just a warning --- go on?	*/
	      }
	  }
	  if (extract_files) {
	      if (fs_extract(&fs,argv[i],1) != 0) {
		  /* can't extract it...can't write output?  other error?	*/
		  /* error or maybe just a warning --- go on?	*/
	      }
	  }
      }
  }

  if (report) {
      fs_report(&fs,(machineform)?0:(terse)?1:(verbose)?3:2);
  }

  fs_cleanup(&fs);

  /* now the file needs to be spit out in the case of -A or -D	*/

  if (add_files || del_files || compress) {

      if (!no_effect) {
	  if (compress) {
	      fs_compress(&fs);
	  }

	  /* since floppy dirs changed, as well as new data added	*/
	  /*   to the disk, the CRCs need to be recalculated	*/

	  floppy_crc_set(&floppy);

	  if (output_format_id == NULL) {
	      output_format_id = floppy_format_list_select("SVD",TRUE);
	  }
	  (void) floppy_format_dumper(output_format_id,1,&floppy);
      }
  }

  exit(0);
}

usage(char *name, char *default_fs)
{
    char	buffer[500];

  printf("Usage: %s [options] [-F fs-type] [-f INPUT-IMAGE] [FILENAMES...]\n\n",name);

  printf("   Decodes and manages the File System (FS) found\n");
  printf("   on the incoming floppy image.  Will automagically guess\n");
  printf("   the format of the incoming image.\n");
  printf("   Spits out an SVD formatted version of the floppy image\n");
  printf("   after the given operation was performed.  In this way\n");
  printf("   you can string a whole bunch of operations together\n");
  printf("   in a nice long pipe if you wish.\n");

  printf("\n");

  printf("   Automagically guesses the INPUT-IMAGE format\n");
  printf("   of either %s.\n", floppy_format_list_string(buffer,FALSE,FALSE));
  printf("   If it doesn't guess right, use the select option: -s fmt\n");

  printf("\n");

  printf("   Also automagically trys to guess the FS that is on the image.\n");
  printf("   However, since there are numerous FS's, it may not get it\n");
  printf("   right and you'll have to specify the FS type.\n");

  printf("\n");

  printf(" Supported options\n");
  printf("   -f image ---  use the given floppy FS image (\"-\" is stdin)\n");
  printf("   -F type  ---  the type of the filesystem, defaults to \"%s\"\n",default_fs);
  printf("                   (one of: ");
  { 
      int i;

      for (i=0; i < fs_count; i++) {
	  printf("%s%s",fs_types[i],(i==fs_count-1)?")":" ");
      }
      printf("\n");
  }
  printf("   -l     ---  list the types of filesystems available (also used with -v)\n");
  printf("   -s fmt ---  forces a specific input format interpretation; fmt needs to\n");
  printf("               to be one of: %s\n", floppy_format_list_string(buffer,FALSE,FALSE));
  printf("   -r     ---  report on the data in the image (like \"dir\") - implies -n\n");
  printf("   -A     ---  add the given FILENAME(s) to the image\n");
  printf("   -D     ---  delete the given FILENAMES(s) from the image\n");
  printf("   -E     ---  extract the given FILENAMES(s) from the image\n");
  printf("   -v     ---  do everything in a noisy/verbose way\n");
  printf("   -t     ---  be very terse in reporting\n");
  printf("   -n     ---  NO EFFECT option (just like for \"make\")\n");
  printf("               Just checks to see if the operation would be\n");
  printf("               performed appropriately, generating messages\n");
  printf("               and an appropriate return value.\n");
  printf("   -o fmt ---  output is written in the given fmt; fmt must be\n");
  printf("               one of: %s\n", floppy_format_list_string(buffer,TRUE,FALSE));
  printf("               (defaults to: SVD)\n");
  printf("   -m     ---  report in a machine readable form\n");
  printf("   -c     ---  compress the output floppy image\n");
  printf("   -V     ---  print out version number\n");

  printf("\n");

  printf("   Returns:   0 if OK, 1 if error\n");
}

/************************************************************************
 * NAME:	read_image()
 *
 * DESCR:	Read in the floppy image.
 *
 * ARGS:	
 *
 * RETURNS:	0 if things went peachy, something else otherwise.
 *
 * NOTES:	
 ************************************************************************/
int
read_image(int fd, struct floppy *floppy, char *floppy_format, char *prog, char *filename)
{
    if (floppy_format == NULL) {
	char 	*ext = NULL;

	if (filename) {
	    ext = floppy_file_extension(filename);
	}

	floppy_format = floppy_format_guesser(fd, ext);

	if (floppy_format == NULL) {
	    fprintf(stderr,"%s: format not recognized.\n",prog);
	    return(FALSE);
	}
    }

    lseek(fd,(off_t)0,SEEK_SET);

    if (floppy_format_reader(floppy_format,fd,floppy)!=0) {
	fprintf(stderr,"%s:  error reading %s format.\n",prog,floppy_format);
	return(FALSE);
    }

    floppy_sectormap(floppy);		/* fill sector map	*/

    return(TRUE);
}
